<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Like;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Models\Post\Like\PostLikeModel;

final class PostLikeContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function load(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postIdFromReq = self::$request -> findKey(
                PostLikeTable::LIKED_POST_ID,
                RequestTable::PAYLOAD,
                PostLikeTable::TABLE_NAME
            );

            $offset = self::$request -> findKeyOffset(PostLikeTable::ID, PostLikeTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | check if post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | check if post publisher is available
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $postLikeTableQuery = (new Query()) -> from(PostLikeTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select where post id is matched
            |--------------------------------------------------------------------------
            */

            $postLikeTableQuery -> where(PostLikeTable::LIKED_POST_ID, $postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $postLikeTableQuery -> greaterThan(PostLikeTable::ID, $offset);
            }
            else
            {
                $postLikeTableQuery -> lessThan(PostLikeTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $postLikeTableQuery
                -> orderByDesc(PostLikeTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_POST_LIKE));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $postLikeBeans = $postLikeTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($postLikeBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(PostLikeTable::getTableName(), $postLikeBeans, function (PostLikeModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | list of users(who's post likes are selected)
                |--------------------------------------------------------------------------
                */

                self::addDependency(UserTable::getTableName(), $model -> getLikedByUserId());
            });

            /*
            |--------------------------------------------------------------------------
            | additional data | follow maps
            | -------------------------------------------------------------------------
            | help build follow button next to user tile
            |--------------------------------------------------------------------------
            */

            $usersContainer = self::$dataDock -> getContainer(UserTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                UserFollowTable::getTableName(),
                array(
                    UserFollowTable::FOLLOWED_USER_ID    => $usersContainer -> getIds(),
                    UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );
        });
    }
}
